function faults(filename)
%FAULTS Function to perform fault current calculations.
% Function faults reads a data set describing a power system
% from a disk file, and performs a fault current analysis on it.  
% The only argument is the name of the input file.  
%
% There are six types of lines in the input file.  The "SYSTEM" 
% line specifies the name of the power system and the base 
% apparent power of the power system in MVA.  Its form is:
%
% SYSTEM name baseMVA 
%
% where
%   name    = The name of the power system
%   baseMVA = The base MVA of the power system
%
% The "BUS" line specifies the characteristics of a bus on the
% power system.  Its form is:
%
% BUS name type volts 
%
% where
%   name   = The name of the bus
%   type   = The type of the bus, one of PQ, PV, or SL
%   Vbus   = The pre-fault voltage at the bus
%
% The "LINE" line specifies the characteristics of a transmission
% line on the power system.  Note that the impedance of the 
% transformers in series with the transmission line should also
% be lumped into these terms.  Its form is:
%
% LINE from to Rse Xse Gsh Bsh X0 Vis
%
% where
%   from   = The name of the "from" bus
%   to     = The type of the "to" bus
%   Rse    = Per-unit series resistance
%   Xse    = Per-unit series reactance
%   Gsh    = Per-unit shunt conductance
%   Bsh    = Per-unit shunt susceptance
%   X0     = Per-unit series zero-sequence reactance
%   Vis    = Zero-sequence visibility flag:
%            0 -- No visible from either bus
%            1 -- Visible from "from" bus
%            2 -- Visible from "to" bus
%            3 -- Visible from both busses
%
% The "GENERATOR" line specifies the characteristics of a 
% generator on the power system.  Its form is:
%
% GENERATOR bus R Xs Xp Xpp X2 X0
%
% where
%   bus    = The bus where the generator is attached
%   R      = Per-unit resistance
%   Xs     = Per-unit synchronous reactance
%   Xp     = Per-unit transient reactance
%   Xpp    = Per-unit subtransient reactance
%   X2     = Per-unit negative sequence reactance
%   X0     = Per-unit zero sequence reactance
%
% The "MOTOR" line specifies the characteristics of a 
% synchronous motor on the power system.  Its form is:
%
% MOTOR bus R Xs Xp Xpp X2 X0
%
% where
%   bus    = The bus where the motor is attached
%   R      = Per-unit resistance
%   Xs     = Per-unit synchronous reactance
%   Xp     = Per-unit transient reactance
%   Xpp    = Per-unit subtransient reactance
%   X2     = Per-unit negative sequence reactance
%   X0     = Per-unit zero sequence reactance
%
% The "FAULT" line specifies the type of fault current
% calculation to perform.  Its form is:
%
% FAULT bus calc_type calc_time
%
% where
%   bus       = The name of the bus with the fault
%   calc_type = The type of calculation to perform
%                 3P  -- Symmetric three-phase fault 
%                 LG  -- Single line-to-ground fault
%                 LL  -- Line-to-line fault
%                 LLG -- Line-line-ground fault
%   calc_time = The time period for the calculation:
%                 0   -- All
%                 1   -- Subtransient
%                 2   -- Transient
%                 3   -- Steady state
%
% The lines should appear in the order:
%  
%  SYSTEM ...
%  BUS ...
%  LINE ...
%  GENERATOR ...
%  MOTOR ...
%  FAULT ...
 
%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    01/12/00    S. J. Chapman        Original code

% Get the power system data
[bus, line, system, gen, mot, fault] = read_system2(filename);

% Solve this system
if fault.calc_time > 0

   % Build Ybus
   [Ybus1, Ybus2, Ybus0] = build_ybus2(bus, line, gen, mot, fault);

   % Build positive-sequence Zbus
   Zbus1 = inv(Ybus1);

   % Build negative-sequence Zbus
   if ~strcmpi(fault.type,'3P')
      Zbus2 = inv(Ybus2);
   else
      Zbus2 = zeros(size(Ybus2));
   end

   % Build zero-sequence Zbus
   if strcmpi(fault.type,'SLG') | strcmpi(fault.type,'LLG')
      Zbus0 = inv(Ybus0);
   else
      Zbus0 = zeros(size(Ybus0));
   end

   % Solve system for the specified type of current
   bus = solve_system2(bus, Zbus1, Zbus2, Zbus0, fault);

   % Display results
   if strcmpi(fault.type,'3P')
      report_system2(1, bus, line, system, Ybus1, Zbus1, gen, mot, fault);
   else
      report_system3(1, bus, line, system, Ybus1, Ybus2, Ybus0, ...
                     Zbus1, Zbus2, Zbus0, gen, mot, fault);
   end
   
else

   for ii = 1:3
      
      % Set calculation time
      fault.calc_time = ii;
      
      % Build Ybus
      [Ybus1, Ybus2, Ybus0] = build_ybus2(bus, line, gen, mot, fault);

      % Build positive-sequence Zbus
      Zbus1 = inv(Ybus1);

      % Build negative-sequence Zbus
      if ~strcmpi(fault.type,'3P')
         Zbus2 = inv(Ybus2);
      else
         Zbus2 = zeros(size(Ybus2));
      end

      % Build zero-sequence Zbus
      if strcmpi(fault.type,'SLG') | strcmpi(fault.type,'LLG')
         Zbus0 = inv(Ybus0);
      else
         Zbus0 = zeros(size(Ybus0));
      end

      % Solve system for the specified type of current
      bus = solve_system2(bus, Zbus1, Zbus2, Zbus0, fault);

      % Display results
      if strcmpi(fault.type,'3P')
         report_system2(1, bus, line, system, Ybus1, Zbus1, gen, mot, fault);
      else
         report_system3(1, bus, line, system, Ybus1, Ybus2, Ybus0, ...
                        Zbus1, Zbus2, Zbus0, gen, mot, fault);
      end
   
   end
end

%================================================================
%================================================================
function [bus, line, system, gen, mot, fault] = read_system2(filename)
%READ_SYSTEM Read a power system from disk.
% Function read_system reads a data set describing a power system
% from a disk file.  
%
% There are six types of lines in the input file.  The "SYSTEM" 
% line specifies the name of the power system and the base 
% apparent power of the power system in MVA.  Its form is:
%
% SYSTEM name baseMVA 
%
% where
%   name    = The name of the power system
%   baseMVA = The base MVA of the power system
%
% The "BUS" line specifies the characteristics of a bus on the
% power system.  Its form is:
%
% BUS name type volts 
%
% where
%   name   = The name of the bus
%   type   = The type of the bus, one of PQ, PV, or SL
%   Vbus   = The pre-fault voltage at the bus
%
% The "LINE" line specifies the characteristics of a transmission
% line on the power system.  Note that the impedance of the 
% transformers in series with the transmission line should also
% be lumped into these terms.  Its form is:
%
% LINE from to Rse Xse Gsh Bsh X0 Vis
%
% where
%   from   = The name of the "from" bus
%   to     = The type of the "to" bus
%   Rse    = Per-unit series resistance
%   Xse    = Per-unit series reactance
%   Gsh    = Per-unit shunt conductance
%   Bsh    = Per-unit shunt susceptance
%   X0     = Per-unit series zero-sequence reactance
%   Vis    = Zero-sequence visibility flag:
%            0 -- No visible from either bus
%            1 -- Visible from "from" bus
%            2 -- Visible from "to" bus
%            3 -- Visible from both busses
%
% The "GENERATOR" line specifies the characteristics of a 
% generator on the power system.  Its form is:
%
% GENERATOR bus R Xs Xp Xpp X2 X0
%
% where
%   bus    = The bus where the generator is attached
%   R      = Per-unit resistance
%   Xs     = Per-unit synchronous reactance
%   Xp     = Per-unit transient reactance
%   Xpp    = Per-unit subtransient reactance
%   X2     = Per-unit negative sequence reactance
%   X0     = Per-unit zero sequence reactance
%
% The "MOTOR" line specifies the characteristics of a 
% synchronous motor on the power system.  Its form is:
%
% MOTOR bus R Xs Xp Xpp X2 X0
%
% where
%   bus    = The bus where the motor is attached
%   R      = Per-unit resistance
%   Xs     = Per-unit synchronous reactance
%   Xp     = Per-unit transient reactance
%   Xpp    = Per-unit subtransient reactance
%   X2     = Per-unit negative sequence reactance
%   X0     = Per-unit zero sequence reactance
%
% The "FAULT" line specifies the type of fault current
% calculation to perform.  Its form is:
%
% FAULT bus calc_type calc_time
%
% where
%   bus       = The name of the bus with the fault
%   calc_type = The type of calculation to perform
%                 3P  -- Symmetric three-phase fault 
%                 LG  -- Single line-to-ground fault
%                 LL  -- Line-to-line fault
%                 LLG -- Line-line-ground fault
%   calc_time = The time period for the calculation:
%                 0   -- All
%                 1   -- Subtransient
%                 2   -- Transient
%                 3   -- Steady state
%
% The lines should appear in the order:
%  
%  SYSTEM ...
%  BUS ...
%  LINE ...
%  GENERATOR ...
%  MOTOR ...
%  FAULT ...
 
%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    01/12/00    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(1,1,nargin);
error(msg);

% Initialise counters
n_system  = 0;          % Number of SYSTEM cards
n_bus     = 0;          % Number of BUS cards
n_gen     = 0;          % Number of GENERATOR cards
n_lines   = 0;          % Number of LINE cards
n_gen     = 0;          % Number of MOTOR cards
n_mot     = 0;          % Number of INVALID cards
n_type    = 0;          % Number of TYPE cards
n_bad     = 0;          % Number of BAD lines
n_comment = 0;          % Number of comment lines
i_line    = 0;          % Current line number
mot       = [];         % Initialize motor array

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Open input file
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[fid,message] = fopen(filename,'r');  % FID = -1 for failure.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check for error
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if fid < 0
   str = ['ERROR: Can''t find system file: ' filename];
   error(str);

else

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % File open OK, so read lines.
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   while feof(fid) == 0
   
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Get next line
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      data = fgetl(fid);
      i_line = i_line + 1;
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Extract keyword
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      maxlen = min( [ 9 length(data) ] );
      keyword = data(1:maxlen);
   
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Determine the type of the line
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      if strncmpi(keyword,'SYSTEM',6) == 1
      
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a SYSTEM card
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_system = n_system + 1;
         
         % Get system name
         blanks = findstr(data,' ');
         test = diff(blanks) > 1;
         for ii = 1:length(test)
            if test(ii) > 0
               system.name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get base MVA
         ii = blanks(ii+1);
         temp = sscanf(data(ii:length(data)),'%g');
         system.baseMVA = temp(1);
         
         % Voltage tolerance
         if length(temp) > 1
            system.v_tol = temp(2);
         else
            system.v_tol = 0.05;
         end
         
      elseif strncmpi(keyword,'BUS',3) == 1
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a BUS card
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_bus = n_bus + 1;
         
         % Confirm that a SYSTEM card has preceeded this card
         if n_system == 0
            error (['ERROR: A SYSTEM card must precede any BUS cards!']);
         end

         % Get bus name
         blanks = findstr(data,' ');
         test = diff(blanks) > 1;
         for ii = 1:length(test)
            if test(ii) > 0
               bus(n_bus).name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get voltage
         ii = blanks(ii+1);
         temp = sscanf(data(ii:length(data)),'%g');
         bus(n_bus).Vbus = temp(1);
         
      elseif strncmpi(keyword,'LINE',4) == 1
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a LINE card
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_lines = n_lines + 1;
         
         % Confirm that a SYSTEM card has preceeded this card
         if n_system == 0
            error (['ERROR: A SYSTEM card must precede any LINE cards!']);
         end

         % Get "from" bus name
         blanks = findstr(data,' ');
         test = diff(blanks) > 1;
         for ii = 1:length(test)
            if test(ii) > 0
               line(n_lines).from_name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get "to" bus name
         for ii = ii+1:length(test)
            if test(ii) > 0
               line(n_lines).to_name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get numeric values
         ii = blanks(ii+1);
         temp = sscanf(data(ii:length(data)),'%g');

         % Get values
         line(n_lines).Rse    = temp(1);
         line(n_lines).Xse    = temp(2);
         line(n_lines).Gsh    = temp(3);
         line(n_lines).Bsh    = temp(4);
         line(n_lines).X0     = temp(5);
         line(n_lines).vis    = temp(6);

      elseif strncmpi(keyword,'GENERATOR',9) == 1
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a GENERATOR card
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_gen = n_gen + 1;
         
         % Get bus name
         blanks = findstr(data,' ');
         test = diff(blanks) > 1;
         for ii = 1:length(test)
            if test(ii) > 0
               gen(n_gen).bus_name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get numeric values
         ii = blanks(ii+1);
         temp = sscanf(data(ii:length(data)),'%g');

         % Get values
         gen(n_gen).R      = temp(1);
         gen(n_gen).Xs     = temp(2);
         gen(n_gen).Xp     = temp(3);
         gen(n_gen).Xpp    = temp(4);
         gen(n_gen).X2     = temp(5);
         gen(n_gen).X0     = temp(6);

      elseif strncmpi(keyword,'MOTOR',5) == 1
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a MOTOR card
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_mot = n_mot + 1;
         
         % Get bus name
         blanks = findstr(data,' ');
         test = diff(blanks) > 1;
         for ii = 1:length(test)
            if test(ii) > 0
               mot(n_mot).bus_name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get numeric values
         ii = blanks(ii+1);
         temp = sscanf(data(ii:length(data)),'%g');

         % Get values
         mot(n_mot).R      = temp(1);
         mot(n_mot).Xs     = temp(2);
         mot(n_mot).Xp     = temp(3);
         mot(n_mot).Xpp    = temp(4);
         mot(n_gen).X2     = temp(5);
         mot(n_gen).X0     = temp(6);

      elseif strncmpi(keyword,'FAULT',5) == 1
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a FAULT card
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_type = n_type + 1;
         
         % Get faulted bus name
         blanks = findstr(data,' ');
         test = diff(blanks) > 1;
         for ii = 1:length(test)
            if test(ii) > 0
               fault.name = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         
         % Get fault type
         fault.type = ' ';
         for ii = ii+1:length(test)
            if test(ii) > 0
               fault.type = data(blanks(ii)+1:blanks(ii+1)-1);
               break;
            end
         end
         %if fault.type == ' '
         %   fault.type = data(blanks(length(blanks))+1:length(data));
         %end
         
         % Get numeric values
         ii = blanks(ii+1);
         temp = sscanf(data(ii:length(data)),'%g');

         % Get values
         fault.calc_time = temp(1);

      elseif isempty(keyword)
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a null line--do nothing
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         %n_comment = n_comment + 1;
         
      elseif keyword(1:1) == '%'
   
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is a comment line
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_comment = n_comment + 1;
         
      else
      
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         % This is an invalid line
         %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         n_bad = n_bad + 1;
         if ischar(data)
            disp(['WARNING: Invalid line ' int2str(i_line) ':  ' data]);
         end
         
      end
   end

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % Now test input data for consistency.
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   
   % Initialize line count for each bus.
   for ii = 1:n_bus
      bus(ii).n_lines = 0;
   end
   
   % See where each line goes.
   for ii = 1:n_lines 
      line(ii).from = 0;
      line(ii).to = 0;
      
      % Check for line terminations
      for jj = 1:n_bus
         if strcmpi(line(ii).from_name, bus(jj).name)
            bus(jj).n_lines = bus(jj).n_lines + 1;
            line(ii).from   = jj;
         end
         if strcmpi(line(ii).to_name, bus(jj).name)
            bus(jj).n_lines = bus(jj).n_lines + 1;
            line(ii).to     = jj;
         end
      end
   end
   
   % Initialize bus number for each generator
   for ii = 1:n_gen
      gen(ii).bus_no = 0;
   end
   
   % Check for generator bus name match
   for ii = 1:n_gen
      for jj = 1:n_bus
         if strcmpi(gen(ii).bus_name, bus(jj).name)
            gen(ii).bus_no = jj;
         end
      end
   end
   
   % Initialize bus number for each motor
   for ii = 1:n_mot
      mot(ii).bus_no = 0;
   end
   
   % Check for motor bus name match
   for ii = 1:n_mot
      for jj = 1:n_bus
         if strcmpi(mot(ii).bus_name, bus(jj).name)
            mot(ii).bus_no = jj;
         end
      end
   end

   % Check for fault bus number
   fault.bus_no = 0;
   for jj = 1:n_bus
      if strcmpi(fault.name, bus(jj).name)
         fault.bus_no = jj;
      end
   end
   
   % Check to see that one and only one SYSTEM card
   % was present.
   error_count = 0;
   if n_system == 0
      error_count = error_count + 1;
      disp (['ERROR: No SYSTEM card supplied!']);
   end
   if n_system > 1
      error_count = error_count + 1;
      disp (['ERROR: Too many SYSTEM cards supplied!']);
   end
   
   % Check to see that one and only one fault TYPE card
   % was present.
   error_count = 0;
   if n_type == 0
      error_count = error_count + 1;
      disp (['ERROR: No TYPE card supplied!']);
   end
   if n_type > 1
      error_count = error_count + 1;
      disp (['ERROR: Too many TYPE cards supplied!']);
   end
   
   % Check to see that there are no isolated busses.
   for ii = 1:n_bus
      if bus(ii).n_lines <= 0
         error_count = error_count + 1;
         disp (['ERROR: Isolated bus: ' bus(ii).name]);
      end
   end
   
   % Now check for lines with invalid bus names.
   for ii = 1:n_lines
      if line(ii).from <= 0
         error_count = error_count + 1;
         str = ['ERROR: Invalid from bus on line ' num2str(ii) ...
                ': ' line(ii).from_name];
         disp(str);
      end
      if line(ii).to <= 0
         error_count = error_count + 1;
         str = ['ERROR: Invalid  to  bus on line ' num2str(ii) ...
                ': ' line(ii).to_name];
         disp(str);
      end
   end

   % Now check for GENERATORs with invalid bus names.
   for ii = 1:n_gen
      if gen(ii).bus_no <= 0
         error_count = error_count + 1;
         str = ['ERROR: Invalid bus on GENERATOR ' num2str(ii) ...
                ': ' gen(ii).bus_name];
         disp(str);
      end
   end

   % Now check for MOTORs with invalid bus names.
   for ii = 1:n_mot
      if mot(ii).bus_no <= 0
         error_count = error_count + 1;
         str = ['ERROR: Invalid bus on MOTOR ' num2str(ii) ...
                ': ' mot(ii).bus_name];
         disp(str);
      end
   end

   % Now check for FAULT with invalid bus names.
   if fault.bus_no <= 0
      error_count = error_count + 1;
      str = ['ERROR: Invalid bus on FAULT line ' num2str(ii) ...
             ': ' fault.name];
      disp(str);
   end

   % Now check for invalid FAULT type.
   if strncmpi(fault.type,'3P',2)
      fault.type_str = 'Symmetrical Three-Phase Fault';
   elseif strncmpi(fault.type,'SLG',2)
      fault.type_str = 'Single Line-to-Ground Fault';
   elseif strncmpi(fault.type,'LL',2)
      fault.type_str = 'Line-to-Line Fault';
   elseif strncmpi(fault.type,'DLG',3)
      fault.type_str = 'Double Line-to-Ground Fault';
   else
      error_count = error_count + 1;
      str = ['ERROR: Invalid type on FAULT line: ' ...
             fault.type];
      disp(str);
   end

   % Now check for invalid FAULT calculation time.
   if fault.calc_time == 0
      fault.calc_str = 'All';
   elseif fault.calc_time == 1
      fault.calc_str = 'Subtransient';
   elseif fault.calc_time == 2
      fault.calc_str = 'Transient';
   elseif fault.calc_time == 3
      fault.calc_str = 'Steady-State';
   else
      error_count = error_count + 1;
      str = ['ERROR: Invalid calc time on FAULT line: ' ...
             num2str(fault.calc_time)];
      disp(str);
   end

   % If there were errors, abort with error message.
   if error_count > 0
      disp([int2str(error_count) ' errors total!']);
   end
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % Write out data summary
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   fprintf('Input summary statistics:\n' );
   fprintf('%4d lines in system file\n', i_line );
   fprintf('%4d SYSTEM lines\n', n_system );
   fprintf('%4d BUS lines\n', n_bus );
   fprintf('%4d LINE lines\n', n_lines );
   fprintf('%4d GENERATOR lines\n', n_gen );
   fprintf('%4d MOTOR lines\n', n_mot );
   fprintf('%4d TYPE lines\n', n_type );
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % Close file
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   fclose(fid);
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % Abort on errors
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if error_count > 0
      error(['Job aborted due to input errors.']);
   end
   

end

%================================================================
%================================================================
function [ybus1, ybus2, ybus0] = build_ybus2(bus, line, gen, mot, fault)
%BUILD_YBUS2 Build the bus admittance matrix for fault current calcs
% Function build_ybus2 builds a bus admittance matrix from an input
% structure of lines, generators, and motors.  Each line stretches 
% between two busses, and it characterized by a per-unit series impedance 
% and a per-unit shunt admittance.  Each array element in the "line" 
% array contains the following fields:
%
% from_name     Name of "from" bus
% from          Number of "from" bus
% to_name       Name of "to" bus
% to            Number of "to" bus
% Rse           Series resistance, pu
% Xse           Series reactance, pu
% Gsh           Shunt conductance, pu
% Bsh           Shunt susceptance, pu
% X0            Zero-sequence series reactance, pu
% vis           Visibility flag for zero-sequence reactance:
%                 0 -- No visible from either bus
%                 1 -- Visible from "from" bus
%                 2 -- Visible from "to" bus
%                 3 -- Visible from both busses
%
% Each motor and generator array element has the following fields:
%
% bus_name      Name of bus where connected
% bus_no        Number of bus where connected
% R             Series resistance, pu
% Xs            Steady-state synchronous reactance, pu
% Xp            Transient reactance, pu
% Xpp           Subtransient reactance, pu
% X2            Negative sequence reactance, pu
% X0            Zero sequence reactance, pu
 
%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    01/12/00    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(5,5,nargin);
error(msg);

% Get the number of busses included in the system.
n_bus = length(bus);

% Create ybus 
ybus1 = zeros(n_bus,n_bus);
ybus2 = zeros(n_bus,n_bus);
ybus0 = zeros(n_bus,n_bus);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Now build the positive-sequence bus admittance matrix
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for ii = 1:length(line)

   % Get indices
   fr = line(ii).from;
   to = line(ii).to;

   % Convert series impedance to a series admittance, and
   % conductance and susceptance to a shunt admittance.
   Yse = 1.0 / ( line(ii).Rse + j*line(ii).Xse );
   Ysh = line(ii).Gsh + j*line(ii).Bsh;

   % Diagonal terms
   ybus1(fr,fr) = ybus1(fr,fr) + Yse;
   ybus1(to,to) = ybus1(to,to) + Yse;
   
   % Off-diagonal terms
   ybus1(fr,to) = ybus1(fr,to) - Yse;
   ybus1(to,fr) = ybus1(to,fr) - Yse;

   % Shunt admittance.  Add half of the shunt admittance
   % to the nodes at each end of the transmission line
   % (pi model).
   ybus1(fr,fr) = ybus1(fr,fr) + Ysh/2;
   ybus1(to,to) = ybus1(to,to) + Ysh/2;

end

% Now add generators to Ybus
for ii = 1:length(gen)

   % Get index
   i1 = gen(ii).bus_no;
   
   % Get the proper reactance
   if fault.calc_time == 1
      X = gen(ii).Xpp;
   elseif fault.calc_time == 2
      X = gen(ii).Xp;
   elseif fault.calc_time == 3
      X = gen(ii).Xs;
   end
   
   % Convert series impedance to series admittance
   Yse = 1.0 / ( gen(ii).R + j*X  );

   % Diagonal terms only
   ybus1(i1,i1) = ybus1(i1,i1) + Yse;

end

% Now add motors to Ybus
for ii = 1:length(mot)

   % Get index
   i1 = mot(ii).bus_no;
   
   % Get the proper reactance
   if fault.calc_time == 1
      X = mot(ii).Xpp;
   elseif fault.calc_time == 2
      X = mot(ii).Xp;
   elseif fault.calc_time == 3
      X = mot(ii).Xs;
   end
   
   % Convert series impedance to series admittance.
   % Note that motors do not contribute to the 
   % fault current under steady-state conditions!
   if fault.calc_time ~= 3
      Yse = 1.0 / ( mot(ii).R + j*X );
   else
      Yse = 0.0;
   end

   % Diagonal terms only
   ybus1(i1,i1) = ybus1(i1,i1) + Yse;
   
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Now build the negative-sequence bus admittance matrix
% if it is needed (all faults except 3-phase faults).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~strcmpi(fault.type,'3P')
   for ii = 1:length(line)

      % Get indices
      fr = line(ii).from;
      to = line(ii).to;

      % Convert series impedance to a series admittance, and
      % conductance and susceptance to a shunt admittance.
      Yse = 1.0 / ( line(ii).Rse + j*line(ii).Xse );
      Ysh = line(ii).Gsh + j*line(ii).Bsh;

      % Diagonal terms
      ybus2(fr,fr) = ybus2(fr,fr) + Yse;
      ybus2(to,to) = ybus2(to,to) + Yse;
      
      % Off-diagonal terms
      ybus2(fr,to) = ybus2(fr,to) - Yse;
      ybus2(to,fr) = ybus2(to,fr) - Yse;

      % Shunt admittance.  Add half of the shunt admittance
      % to the nodes at each end of the transmission line
      % (pi model).
      ybus2(fr,fr) = ybus2(fr,fr) + Ysh/2;
      ybus2(to,to) = ybus2(to,to) + Ysh/2;

    end

    % Now add generators to Ybus
    for ii = 1:length(gen)

      % Get index
      i1 = gen(ii).bus_no;
      
      % Convert series impedance to series admittance
      Y2 = 1.0 / ( gen(ii).R + j*gen(ii).X2 );

      % Diagonal terms only
      ybus2(i1,i1) = ybus2(i1,i1) + Y2;
      
   end

   % Now add motors to Ybus
   for ii = 1:length(mot)

      % Get index
      i1 = mot(ii).bus_no;
      
      % Convert series impedance to series admittance
      Y0  = 1.0 / ( mot(ii).R + j*mot(ii).X0 );

      % Diagonal terms only
      ybus2(i1,i1) = ybus2(i1,i1) + Y2;
      
   end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Now build the zero-sequence bus admittance matrix
% if it is needed (SLG and LLG).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if strcmpi(fault.type,'SLG') | strcmpi(fault.type,'LLG')
   for ii = 1:length(line)

      % Get indices
      fr = line(ii).from;
      to = line(ii).to;

      % Convert series impedance to a series admittance, and
      % conductance and susceptance to a shunt admittance.
      Y0  = 1.0 / ( line(ii).Rse + j*line(ii).X0 );
      Ysh = line(ii).Gsh + j*line(ii).Bsh;
      
      % =========== On-diagonal terms =============
      % Handle zero-sequence cases.  If vis = 1 or 3, this
      % admittance is visible from the sending end of the
      % line.
      if line(ii).vis == 1 | line(ii).vis == 3
         ybus0(fr,fr) = ybus0(fr,fr) + Y0;
      end

      % Handle zero-sequence cases.  If vis = 2 or 3, this
      % admittance is visible from the receiving end of the
      % line.
      if line(ii).vis == 2 | line(ii).vis == 3
         ybus0(to,to) = ybus0(to,to) + Y0;
      end

      % =========== Off-diagonal terms =============
      % Handle zero-sequence cases.  If vis = 3, this
      % admittance stretches between two nodes.
      if line(ii).vis == 3
         ybus0(fr,to) = ybus0(fr,to) - Y0;
         ybus0(to,fr) = ybus0(to,fr) - Y0;
      end
      
      % =========== Shunt admittance =============
      % Handle zero-sequence cases.  If vis = 1 or 3, this
      % admittance is visible from the sending end of the
      % line.
      if line(ii).vis == 1 | line(ii).vis == 3
         ybus0(fr,fr) = ybus0(fr,fr) + Ysh/2;
      end

      % Handle zero-sequence cases.  If vis = 2 or 3, this
      % admittance is visible from the receiving end of the
      % line.
      if line(ii).vis == 2 | line(ii).vis == 3
         ybus0(to,to) = ybus0(to,to) + Ysh/2;
      end

    end

    % Now add generators to Ybus
    for ii = 1:length(gen)

      % Get index
      i1 = gen(ii).bus_no;
      
      % Convert series impedance to series admittance
      Y0 = 1.0 / ( gen(ii).R + j*gen(ii).X0 );

      % Diagonal terms only
      ybus0(i1,i1) = ybus0(i1,i1) + Y0;
      
   end

   % Now add motors to Ybus
   for ii = 1:length(mot)

      % Get index
      i1 = mot(ii).bus_no;
      
      % Convert series impedance to series admittance
      Y0  = 1.0 / ( mot(ii).R + j*mot(ii).X0 );

      % Diagonal terms only
      ybus0(i1,i1) = ybus0(i1,i1) + Y0;
      
   end
end

%================================================================
%================================================================
function bus = solve_system2(bus, Zbus1, Zbus2, Zbus0, fault)
%SOLVE_SYSTEM2 Solve for the fault currents and bus voltages in the system
% Function solve_system2 solves for the fault currents and bus voltage 
% in the power system.
 
%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    01/12/00    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(5,5,nargin);
error(msg);

% Define constants
a  = -0.5 + j*0.86602540378444;
a2 = -0.5 - j*0.86602540378444;

% First determine the type of fault to solve for.
if strncmpi(fault.type,'3P',2)

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   %
   % This is a symmetrical three-phase fault.
   % Calculate fault currents.
   %
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   for ii = 1:length(bus)
   
      % Calculate fault current for specified bus
      if fault.bus_no == ii
   
         % Calculate fault current components
         bus(ii).ia1 = bus(ii).Vbus / Zbus1(ii,ii);
         bus(ii).ia2 = 0;
         bus(ii).ia0 = 0;
      
         % Total fault current in each phase: i(1) is phase a, i(2) is
         % phase b, and i(3) is phase c.
         bus(ii).i(1) = bus(ii).ia1      + bus(ii).ia2      + bus(ii).ia0;
         bus(ii).i(2) = a2 * bus(ii).ia1 + a * bus(ii).ia2  + bus(ii).ia0;
         bus(ii).i(3) = a * bus(ii).ia1  + a2 * bus(ii).ia2 + bus(ii).ia0;

      else
      
         % Calculate fault current components
         bus(ii).ia1 = 0;
         bus(ii).ia2 = 0;
         bus(ii).ia0 = 0;
      
         % Total fault current
         bus(ii).i(1) = 0;
         bus(ii).i(2) = 0;
         bus(ii).i(3) = 0;

      end
   end

   % Calculate bus voltages at various nodes
   ii = fault.bus_no;
   for jj = 1:length(bus)
   
      % Calculate bus voltage components
      bus(jj).va1 = bus(jj).Vbus - bus(ii).ia1 * Zbus1(jj,ii);
      bus(jj).va2 =              - bus(ii).ia2 * Zbus2(jj,ii);
      bus(jj).va0 =              - bus(ii).ia0 * Zbus0(jj,ii);
   
      % Total bus voltages in each phase: v(1) is phase a, v(2) is
      % phase b, and v(3) is phase c.
      bus(jj).v(1) = bus(jj).va1      + bus(jj).va2      + bus(jj).va0;
      bus(jj).v(2) = a2 * bus(jj).va1 + a * bus(jj).va2  + bus(jj).va0;
      bus(jj).v(3) = a * bus(jj).va1  + a2 * bus(jj).va2 + bus(jj).va0;

   end

elseif strncmpi(fault.type,'SLG',2)

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   %
   % This is an unsymmetrical single line-to-ground fault.
   % Calculate fault currents.
   %
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   for ii = 1:length(bus)
   
      % Calculate fault current and voltages for specified bus
      if fault.bus_no == ii
   
         % Calculate fault current components
         bus(ii).ia1 = bus(ii).Vbus ...
                     / (Zbus1(ii,ii) + Zbus2(ii,ii) + Zbus0(ii,ii));
         bus(ii).ia2 = bus(ii).ia1;
         bus(ii).ia0 = bus(ii).ia1;

         % Total fault current in each phase: i(1) is phase a, i(2) is
         % phase b, and i(3) is phase c.
         bus(ii).i(1) = bus(ii).ia1      + bus(ii).ia2      + bus(ii).ia0;
         bus(ii).i(2) = a2 * bus(ii).ia1 + a * bus(ii).ia2  + bus(ii).ia0;
         bus(ii).i(3) = a * bus(ii).ia1  + a2 * bus(ii).ia2 + bus(ii).ia0;
         
      else
      
         % Calculate fault current components
         bus(ii).ia1 = 0;
         bus(ii).ia2 = 0;
         bus(ii).ia0 = 0;
      
         % Total fault current
         bus(ii).i(1) = 0;
         bus(ii).i(2) = 0;
         bus(ii).i(3) = 0;
      
      end
   end

   % Calculate bus voltages at various nodes
   ii = fault.bus_no;
   for jj = 1:length(bus)
   
      % Calculate bus voltage components
      bus(jj).va1 = bus(jj).Vbus - bus(ii).ia1 * Zbus1(jj,ii);
      bus(jj).va2 =              - bus(ii).ia2 * Zbus2(jj,ii);
      bus(jj).va0 =              - bus(ii).ia0 * Zbus0(jj,ii);
   
      % Total bus voltages in each phase: v(1) is phase a, v(2) is
      % phase b, and v(3) is phase c.
      bus(jj).v(1) = bus(jj).va1      + bus(jj).va2      + bus(jj).va0;
      bus(jj).v(2) = a2 * bus(jj).va1 + a * bus(jj).va2  + bus(jj).va0;
      bus(jj).v(3) = a * bus(jj).va1  + a2 * bus(jj).va2 + bus(jj).va0;

   end

elseif strncmpi(fault.type,'LL',2)

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   %
   % This is an unsymmetrical line-to-line fault.
   % Calculate fault currents.
   %
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   for ii = 1:length(bus)
   
      % Calculate fault current for specified bus
      if fault.bus_no == ii
   
         % Calculate fault current components
         bus(ii).ia1 = bus(ii).Vbus ...
                     / (Zbus1(ii,ii) + Zbus2(ii,ii));
         bus(ii).ia2 = -bus(ii).ia1;
         bus(ii).ia0 = 0;
      
         % Total fault current in each phase: i(1) is phase a, i(2) is
         % phase b, and i(3) is phase c.
         bus(ii).i(1) = bus(ii).ia1      + bus(ii).ia2      + bus(ii).ia0;
         bus(ii).i(2) = a2 * bus(ii).ia1 + a * bus(ii).ia2  + bus(ii).ia0;
         bus(ii).i(3) = a * bus(ii).ia1  + a2 * bus(ii).ia2 + bus(ii).ia0;
         
      else
      
         % Calculate fault current components
         bus(ii).ia1 = 0;
         bus(ii).ia2 = 0;
         bus(ii).ia0 = 0;
      
         % Total fault current
         bus(ii).i(1) = 0;
         bus(ii).i(2) = 0;
         bus(ii).i(3) = 0;
            
      end
   end

   % Calculate bus voltages at various nodes
   ii = fault.bus_no;
   for jj = 1:length(bus)
   
      % Calculate bus voltage components
      bus(jj).va1 = bus(jj).Vbus - bus(ii).ia1 * Zbus1(jj,ii);
      bus(jj).va2 =              - bus(ii).ia2 * Zbus2(jj,ii);
      bus(jj).va0 =              - bus(ii).ia0 * Zbus0(jj,ii);
   
      % Total bus voltages in each phase: v(1) is phase a, v(2) is
      % phase b, and v(3) is phase c.
      bus(jj).v(1) = bus(jj).va1      + bus(jj).va2      + bus(jj).va0;
      bus(jj).v(2) = a2 * bus(jj).va1 + a * bus(jj).va2  + bus(jj).va0;
      bus(jj).v(3) = a * bus(jj).va1  + a2 * bus(jj).va2 + bus(jj).va0;

   end

elseif strncmpi(fault.type,'DLG',2)

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   %
   % This is an unsymmetrical double line-to-ground fault.
   % Calculate fault currents.
   %
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   for ii = 1:length(bus)
   
      % Calculate fault current for specified bus
      if fault.bus_no == ii
   
         % Calculate fault current components
         Z = Zbus1(ii,ii) + Zbus0(ii,ii) * Zbus2(ii,ii) ...
           / ( Zbus0(ii,ii) + Zbus2(ii,ii) );
         bus(ii).ia1 = bus(ii).Vbus / Z;
         bus(ii).ia2 = -bus(ii).ia1 * Zbus0(ii,ii) / ( Zbus0(ii,ii) + Zbus2(ii,ii) );
         bus(ii).ia0 = -bus(ii).ia1 * Zbus2(ii,ii) / ( Zbus0(ii,ii) + Zbus2(ii,ii) );
      
         % Total fault current in each phase: i(1) is phase a, i(2) is
         % phase b, and i(3) is phase c.
         bus(ii).i(1) = bus(ii).ia1      + bus(ii).ia2      + bus(ii).ia0;
         bus(ii).i(2) = a2 * bus(ii).ia1 + a * bus(ii).ia2  + bus(ii).ia0;
         bus(ii).i(3) = a * bus(ii).ia1  + a2 * bus(ii).ia2 + bus(ii).ia0;

      else
      
         % Calculate fault current components
         bus(ii).ia1 = 0;
         bus(ii).ia2 = 0;
         bus(ii).ia0 = 0;
      
         % Total fault current
         bus(ii).i(1) = 0;
         bus(ii).i(2) = 0;
         bus(ii).i(3) = 0;
            
      end
   end

   % Calculate bus voltages at various nodes
   ii = fault.bus_no;
   for jj = 1:length(bus)
   
      % Calculate bus voltage components
      bus(jj).va1 = bus(jj).Vbus - bus(ii).ia1 * Zbus1(jj,ii);
      bus(jj).va2 =              - bus(ii).ia2 * Zbus2(jj,ii);
      bus(jj).va0 =              - bus(ii).ia0 * Zbus0(jj,ii);
   
      % Total bus voltages in each phase: v(1) is phase a, v(2) is
      % phase b, and v(3) is phase c.
      bus(jj).v(1) = bus(jj).va1      + bus(jj).va2      + bus(jj).va0;
      bus(jj).v(2) = a2 * bus(jj).va1 + a * bus(jj).va2  + bus(jj).va0;
      bus(jj).v(3) = a * bus(jj).va1  + a2 * bus(jj).va2 + bus(jj).va0;

   end

end


%================================================================
%================================================================
function report_system2(fid, bus, line, system, ybus, z_bus, ...
                        gen, mot, fault)
%REPORT_SYSTEM2 Write a report of power system fault currents.
% Function report_system2 writes a fault current report to unit 
% "fid".  If this unit is a file, the file must be opened before 
% the function is called.
 
%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    01/12/00    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(9,9,nargin);
error(msg);

% Get number of busses
n_bus = length(bus);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get type of current being displayed
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if fault.calc_time == 1
   fault.calc_str = 'Subtransient';
elseif fault.calc_time == 2
   fault.calc_str = 'Transient';
elseif fault.calc_time == 3
   fault.calc_str = 'Steady-State';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculate and display bus and line quantities
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Set to titles and labels
fprintf('\n');
fprintf('                           Results for Case %s\n\n', ...
   system.name);
fprintf('%s at Bus %s\n', fault.type_str, bus(fault.bus_no).name);
fprintf('Calculating %s Currents\n', fault.calc_str);
fprintf('|====================Bus Information============');
fprintf('|=====Line Information=====|\n');
fprintf('Bus              Volts / angle    Amps / angle  ');
fprintf('|     To  |  Amps / angle  |\n');
fprintf('no.      Name     (pu)   (deg)    (pu)   (deg)  ');
fprintf('|    Bus  |  (pu)   (deg)  |\n');
fprintf('|===============================================');
fprintf('===========================|\n');

% Write out bus information
for ii = 1:n_bus

   % Write bus number
   fprintf('%3d', ii );

   % Write bus name
   fprintf(' %10s', bus(ii).name );
   
   % Write pu voltage 
   [mag, phase] = r2p(bus(ii).v(1));
   fprintf('   %5.3f/', mag );
   fprintf('%7.2f', phase );

   % Write pu fault current 
   [mag, phase] = r2p(bus(ii).i(1));
   fprintf('  %6.3f/', mag );
   fprintf('%7.2f', phase );

   % Calculate the powers flowing out of this bus to other buses
   count = 0;
   for jj = 1:length(line)
   
      if line(jj).from == ii
      
         % This line starts at the current bus.
         % Write the "to" bus name
         count = count + 1;
         if count > 1
            fprintf(' %56s', line(jj).to_name );
         else   
            fprintf(' %10s', line(jj).to_name );
         end
         
         % Calculate the current in line
         kk = line(jj).to;
         il = -ybus(ii,kk) * (bus(ii).v(1) - bus(kk).v(1));
         
         % Display current in line
         [mag, phase] = r2p(il);
         fprintf('   %5.3f/', mag );
         fprintf('%7.2f\n', phase );
         
      elseif line(jj).to == ii
      
         % This line ends at the current bus.
         % Write the "from" bus name
         count = count + 1;
         if count > 1
            fprintf(' %56s', line(jj).from_name );
         else   
            fprintf(' %10s', line(jj).from_name );
         end
         
         % Calculate the current in line
         kk = line(jj).from;
         il = -ybus(ii,kk) * (bus(ii).v(1) - bus(kk).v(1));
         
         % Display current in line
         [mag, phase] = r2p(il);
         fprintf('   %5.3f/', mag );
         fprintf('%7.2f\n', phase );
         
      end
   end
end      

% Write bottom
fprintf('|===============================================');
fprintf('===========================|\n\n');


%================================================================
%================================================================
function report_system3(fid, bus, line, system, Ybus1, Ybus2, ...
                        Ybus0, Zbus1, Zbus2, Zbus0, ...
                        gen, mot, fault)
%REPORT_SYSTEM3 Write a report of power system fault currents.
% Function report_system3 writes a fault current report to unit 
% "fid".  If this unit is a file, the file must be opened before 
% the function is called.
 
%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    01/12/00    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(13,13,nargin);
error(msg);

% Define constants
a  = -0.5 + j*0.86602540378444;
a2 = -0.5 - j*0.86602540378444;
ph(1) = 'a';
ph(2) = 'b';
ph(3) = 'c';

% Get number of busses
n_bus = length(bus);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get type of current being displayed
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if fault.calc_time == 1
   fault.calc_str = 'Subtransient';
elseif fault.calc_time == 2
   fault.calc_str = 'Transient';
elseif fault.calc_time == 3
   fault.calc_str = 'Steady-State';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculate and display bus and line quantities
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Set to titles and labels
fprintf('\n');
fprintf('                           Results for Case %s\n\n', ...
   system.name);
fprintf('%s at Bus %s\n', fault.type_str, bus(fault.bus_no).name);
fprintf('Calculating %s Currents\n', fault.calc_str);
fprintf('|=====================Bus Information=============');
fprintf('|======Line Information======|\n');
fprintf('Bus            P   Volts / angle    Amps / angle  ');
fprintf('|     To  | P  Amps / angle  |\n');
fprintf('no.      Name  h    (pu)   (deg)    (pu)   (deg)  ');
fprintf('|    Bus  | h  (pu)   (deg)  |\n');
fprintf('|=================================================');
fprintf('=============================|\n');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Write out bus information
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for ii = 1:n_bus
   for kk = 1:3

      % Write bus number
      if kk == 1
         fprintf('%3d', ii );
      else 
         fprintf('   ');
      end

      % Write bus name
      if kk == 1
         fprintf(' %10s', bus(ii).name );
      else
         fprintf('           ');
      end
      
      % Write phase 
      fprintf(' %1s', ph(kk) );
      
      % Write pu voltage 
      [mag, phase] = r2p(bus(ii).v(kk));
      fprintf('   %5.3f/', mag );
      fprintf('%7.2f', phase );

      % Write pu fault current 
      [mag, phase] = r2p(bus(ii).i(kk));
      fprintf('  %6.3f/', mag );
      fprintf('%7.2f\n', phase );

   end

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % Write out line information
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % Calculate the current flowing out of this bus to other buses
   for jj = 1:length(line)
   
      if line(jj).from == ii
      
         % Write the "to" bus name
         fprintf(' %58s', line(jj).to_name );
         
         % Calculate the current in line
         kk = line(jj).to;
         ia1 = -Ybus1(ii,kk) * (bus(ii).va1 - bus(kk).va1);
         ia2 = -Ybus2(ii,kk) * (bus(ii).va2 - bus(kk).va2);
         ia0 = -Ybus0(ii,kk) * (bus(ii).va0 - bus(kk).va0);
         ia(1) = ia1      + ia2      + ia0;
         ia(2) = a2 * ia1 + a * ia2  + ia0;
         ia(3) = a * ia1  + a2 * ia2 + ia0;
         
         % Display current in line
         for kk = 1:3
            [mag, phase] = r2p(ia(kk));
            if kk == 1
               fprintf('%4s %5.3f/', ph(kk), mag );
               fprintf('%7.2f\n', phase );
            else
               fprintf('%58s %4s %5.3f/', ' ', ph(kk), mag );
               fprintf('%7.2f\n', phase );
            end
         end
         
      elseif line(jj).to == ii
      
         % Write the "from" bus name
         fprintf(' %58s', line(jj).from_name );
         
         % Calculate the current in line
         kk = line(jj).from;
         ia1 = -Ybus1(ii,kk) * (bus(ii).va1 - bus(kk).va1);
         ia2 = -Ybus2(ii,kk) * (bus(ii).va2 - bus(kk).va2);
         ia0 = -Ybus0(ii,kk) * (bus(ii).va0 - bus(kk).va0);
         ia(1) = ia1      + ia2      + ia0;
         ia(2) = a2 * ia1 + a * ia2  + ia0;
         ia(3) = a * ia1  + a2 * ia2 + ia0;

         % Display current in line
         for kk = 1:3
            [mag, phase] = r2p(ia(kk));
            if kk == 1
               fprintf('%4s %5.3f/', ph(kk), mag );
               fprintf('%7.2f\n', phase );
            else
               fprintf('%58s %4s %5.3f/', ' ', ph(kk), mag );
               fprintf('%7.2f\n', phase );
            end
         end
                  
      end
   end
end      

% Write bottom
fprintf('|=================================================');
fprintf('=============================|\n\n');

%================================================================
%================================================================
function [mag,phase] = r2p(inval)
%R2P Convert a complex number in rectangular format to polar in degrees
% Function R2P converts either a complex number into a complex
% number in polar format, with the angle in degrees.
 
% Define variables:
%   inval     -- Input value in rectangular form
%   mag       -- Magnitude of number
%   phase     -- Phase of number, in degrees

%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    12/12/99    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(1,1,nargin);
error(msg);

% Convert number
mag = abs(inval);
phase = angle(inval) * 180 / pi;

%================================================================
%================================================================
function outval = p2r(mag,phase)
%p2r Convert a complex number in polar format to rectangluar
% Function p2r converts either a complex number in polar format
% with the angle in degrees into a complex number in rectangular
% format.
 
% Define variables:
%   mag       -- Magnitude of number
%   outval    -- Output value in rectangular form
%   phase     -- Phase of number, in degrees

%  Record of revisions:
%      Date       Programmer          Description of change
%      ====       ==========          =====================
%    12/12/99    S. J. Chapman        Original code

% Check for a legal number of input arguments.
msg = nargchk(2,2,nargin);
error(msg);

% Convert number
theta = phase * pi / 180;
outval = mag * ( cos(theta) + j*sin(theta) );
