% M-file: ex11_3.m
% M-file to solve the power-flow problem of Example 11-3.
% This set of equations includes a generator bus.

% Set problem size and initial conditions
n_bus = 4;
swing_bus = 1;
acc_fac = 1.3;

% Specify which busses are generator busses with flags.
% Note that 1 is "true" and 0 is "false" in MATLAB.  In
% this example, Bus 3 is a generator bus, and the others
% are not.
gen_bus = [0 0 1 0];

% Create Y-bus
 Ybus = ...
[ 1.7647-j*7.0588 -0.5882+j*2.3529  0               -1.1765+j*4.7059; ...
 -0.5882+j*2.3529  1.5611-j*6.6290 -0.3846+j*1.9231 -0.5882+j*2.3529; ...
  0               -0.3846+j*1.9231  1.5611-j*6.6290 -1.1765+j*4.7059; ...
 -1.1765+j*4.7059 -0.5882+j*2.3529 -1.1765+j*4.7059  2.9412-j*11.7647 ];
 
% Initialize the real and reactive power supplied to the 
% power system at each bus.  Note that the power at the 
% swing bus doesn't matter, and the reactive power at the
% generator bus will be recomputed dynamically.
P(2) = -0.2;
P(3) =  0.3;
P(4) = -0.2;
Q(2) = -0.15;
Q(3) =  0.0;
Q(4) = -0.10;
         
% Initialize the bus voltages to 1.0 at 0 degrees
for ii = 1:n_bus
   Vbus(ii) = 1;
end

% Set convergence criterion
eps = 0.0001;

% Initialize the iteration counter 
n_iter = 0;

% Set a maximum number of iterations here so that
% the program will not run in an infinite loop if
% it fails to converge to a solution.
for iter = 1:100
   
   % Increment the iteration count
   n_iter = n_iter + 1; 

   % Save old bus voltages for comparison purposes
   Vbus_old = Vbus;

   % Calculate the updated bus voltage
   for ii = 1:n_bus
   
      % Skip the swing bus!
      if ii ~= swing_bus
      
         % If this is a generator bus, update the reactive
         % power estimate.
         if gen_bus(ii) 
            temp = 0;
            for jj = 1:n_bus
               temp = temp + Ybus(ii,jj) * Vbus(jj);
            end
            temp = conj(Vbus(ii)) * temp;
            Q(ii) = -imag(temp);
         end

         % Calculate updated voltage at bus 'ii'.  First, sum
         % up the current contributions at bus 'ii' from all
         % other busses. 
         temp = 0;
         for jj = 1:n_bus
            if ii ~= jj
               temp = temp - Ybus(ii,jj) * Vbus(jj);
            end
         end
         
         % Add in the current injected at this node
         temp = (P(ii) - j*Q(ii)) / conj(Vbus(ii)) + temp;
         
         % Get updated estimate of Vbus at 'ii'
         Vnew = 1/Ybus(ii,ii) * temp;
         
         % Apply an acceleration factor to the new voltage estimate
         Vbus(ii) = Vbus_old(ii) + acc_fac * (Vnew - Vbus_old(ii));
         
         % If this is a generator bus, update the magnitude of the
         % voltage to keep it constant.
         if gen_bus(ii) 
            Vbus(ii) = Vbus(ii) * abs(Vbus_old(ii)) / abs(Vbus(ii));
         end
      end
   end
 
   % Compare the old and new estimate of the voltages.
   % Note that we will compare the real and the imag parts
   % separately, and both must be within tolerances.
   test = 0;
   for ii = 1:n_bus
   
      % Compare real parts
      if abs( real(Vbus(ii)) - real(Vbus_old(ii)) ) > eps
         test = 1;
      end
      
      % Compare imaginary parts
      if abs( imag(Vbus(ii)) - imag(Vbus_old(ii)) ) > eps
         test = 1;
      end
   end
   
   % Did we converge?  If so, get out of the loop.
   if test == 0
      break;
   end
end

% Did we exceed the maximum number of iterations?
if iter == 100
   disp('Max number of iterations exceeded!');
end

% Display results
for ii = 1:n_bus
   [mag, phase] = r2p(Vbus(ii));
   str = ['The voltage at bus ' int2str(ii) ' = ' ... 
          num2str(mag) '/' num2str(phase)];
   disp(str);
end

% Display the number of iterations
str = ['Number of iterations = ' int2str(n_iter) ];
disp(str);
